;;; 
;;; Virtual Disk Drive Project	- Mar 1, 2002 - EJR (on plane to Austin)
;;;
;;; Utility Functions

#include "vd.h"
	
;;;
;;; This routine exports the following symbols
;;; 
   global Mod256	      ; used during alignment
   global ModInverse          ; used during alignment
   global Multiply	      ; simply does a multiply
   global SerPrintNum	      ; puts a number in hex out to serial
   global SerPrintNL	      ; prints out a newline
   global SerPrint1	      ; prints out a value with label (1 byte)
   global SerPrint2	      ; prints out a value with label (2 bytes)
   global XmitWait	      ; returns when able to transmit again

PROG1 CODE
		
;;;*******************************************************************
;;; NAME:	Mod256()
;;;
;;; DESCR:	Divides 256 by the argument in W, and returns the
;;;		remainder.
;;;
;;; ARGS:	W has the divisor
;;;
;;; RETURNS:	W has the remainder.  The quotient is punted.
;;;
;;; NOTES:	- This routine assumes that incoming W is less than 256.
;;;		- WORK1 is used
;;;*******************************************************************
Mod256:
		clrf	WORK1
		subwf	WORK1	;  first subtract

Mod2562:	
		subwf	WORK1
		skpnc		; if rolled over, then prepare return
		goto	Mod2562

		addwf	WORK1,W	; rolled over at this point

		return		; W has the remainder

;;;*******************************************************************
;;; NAME:	ModInverse()
;;;
;;; DESCR:	Divide ARG0 by W, return the the subtraction of the
;;;		divisor and the remainder.  This number is suitable to
;;;		add to the dividend to create a number divisible by
;;;		the divisor.
;;;
;;; ARGS:	W - the divisor (small number)
;;;		ARG0 - the dividend (big number)
;;;
;;; RETURNS:	W - the inverse remainder of the division
;;;
;;; NOTES:	WORK1 is used
;;;*******************************************************************
ModInverse:
		movwf	WORK1	; save a copy of divisor
	
ModI2:	
		subwf	ARG0	; subtract until carry hits
		skpnc
		goto	ModI2

		addwf	ARG0,W	; back up to before failure

		skpnz
		return		; return zero if ARG0 is a multiple of
				;  divisor or was originally zero 

	;; at this point W has remainder, WORK1 has divisor

		subwf	WORK1,W	; find inverse mod and return

		return

;;;*******************************************************************
;;; NAME:	Multiply()
;;;
;;; DESCR:	Multiplies the argument in ARG0 by W.  It is assumed
;;;		in this routine that the result is less than 256.
;;;
;;; ARGS:	ARG0 is the looping argument (header bytes for eg)
;;;		W is the adding arg (number of sectors for eg)
;;;
;;; RETURNS:	result is in W
;;;
;;; NOTES:	- for eg, 18 sectors times 9 header bytes...should loop
;;;		  on the small number...which is probably header bytes.
;;;		- WORK1 is used
;;;		- 0 in the looping arg (ARG0) is interpreted at 256
;;;		- cycles consumed: ARG0 * 3 + 2 (not incl call & return)
;;;*******************************************************************
Multiply:
		clrf	WORK1
Multloop:	
		addwf	WORK1
		decfsz	ARG0
		goto	Multloop
		movf	WORK1,W
		return

;;;*******************************************************************
;;; NAME:	SerPrintNL()
;;;
;;; DESCR:	Prints out a newline.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- uses W
;;;*******************************************************************
SerPrintNL:
		movlw	0x0a		; line feed
		call	XmitWait
		movwf	TXREG

		movlw	0x0d		; carriage return
		call	XmitWait
		movwf	TXREG

		return

;;;**********************************************************************
;;; NAME:	SerPrint1()
;;;
;;; DESCR:	Prints out a given value with a given label.  This is the
;;;		one byte version.
;;;
;;; ARGS:	W has the label, ARG0 has the value
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;**********************************************************************
SerPrint1:
		call	XmitWait
		movwf	TXREG		; label out

		movlw	':'
		call	XmitWait
		movwf	TXREG
	
		movlw	' '
		call	XmitWait
		movwf	TXREG

		movf	ARG0,W
		call	SerPrintNum
	
		call	SerPrintNL
		return

;;;**********************************************************************
;;; NAME:	SerPrint2()
;;;
;;; DESCR:	Prints out a given value with a given label.  This is the
;;;		two byte version.
;;;
;;; ARGS:	W has the label, ARG0 as the high value, ARG1 has the low
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;**********************************************************************
SerPrint2:
		call	XmitWait
		movwf	TXREG		; label out

		movlw	':'
		call	XmitWait
		movwf	TXREG
	
		movlw	' '
		call	XmitWait
		movwf	TXREG

		movf	ARG0,W
		call	SerPrintNum
	
		movf	ARG1,W
		call	SerPrintNum

		call	SerPrintNL	

		return
		
;;;**********************************************************************
;;; NAME:	SerPrintNum	()
;;;
;;; DESCR:	Transmit the given byte (W) out to the serial port in
;;;		hex.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	Uses WORK1
;;;
;;; NOTES:	
;;;**********************************************************************
SerPrintNum:
		movwf	WORK1

		rrf	WORK1
		rrf	WORK1
		rrf	WORK1
		rrf	WORK1		; get upper bits first

		movf	WORK1,W

		call	SerMapDigit

		call	XmitWait
		movwf	TXREG

		rlf	WORK1		; lower bits should be safe
		rlf	WORK1		; (carry bit shouldn't have changed
		rlf	WORK1		;  only adds and subs change it) 
		rlf	WORK1

		movf	WORK1,W

		call	SerMapDigit

		call	XmitWait
		movwf	TXREG

		return


;;;**********************************************************************
;;; NAME:	SerMapDigit()
;;;
;;; DESCR:	Maps out a single hex digit in W.
;;;
;;; ARGS:	uses WORK2
;;;
;;; RETURNS:	Map is in W
;;;
;;; NOTES:	- couldn't get the math right...used brute force
;;;**********************************************************************
SerMapDigit:
		andlw	0x0f	; mask off upper bits
	
		movwf	WORK2
		skpnz
		movlw	'0'

		decf	WORK2
		skpnz
		movlw	'1'

		decf	WORK2
		skpnz
		movlw	'2'

		decf	WORK2
		skpnz
		movlw	'3'

		decf	WORK2
		skpnz
		movlw	'4'

		decf	WORK2
		skpnz
		movlw	'5'

		decf	WORK2
		skpnz
		movlw	'6'

		decf	WORK2
		skpnz
		movlw	'7'

		decf	WORK2
		skpnz
		movlw	'8'

		decf	WORK2
		skpnz
		movlw	'9'

		decf	WORK2
		skpnz
		movlw	'A'

		decf	WORK2
		skpnz
		movlw	'B'

		decf	WORK2
		skpnz
		movlw	'C'

		decf	WORK2
		skpnz
		movlw	'D'

		decf	WORK2
		skpnz
		movlw	'E'

		decf	WORK2
		skpnz
		movlw	'F'
		
		return

;;;*******************************************************************
;;; NAME:	XmitWait()
;;;
;;; DESCR:	Returns only when the serial xmit buffer is available
;;;		for transmitting another byte.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	Often, when debugging, the body of this routine is
;;;		commented out.
;;;*******************************************************************
XmitWait:
	
	btfss	PIR1,TXIF	; Check for xmit buffer clear
	goto	XmitWait	; spin until it is
	return

;;; ---------------------------------------------------------------------------
	
	END